/************************************************************************/
/*									*/
/*	LIBSGL3D.C				(C) 1/94 Ch. Muenkel	*/
/*	Three-dimensional extensions to the (basic) SGL library		*/
/*									*/
/************************************************************************/
/*									*/
/*	Simple Graphics Library	- 3D					*/
/*	An easy to use interface to the X11 window system		*/
/*      This is the wireframe 3D extension to the SGL library           */
/*									*/
/************************************************************************/
/*									*/
/*	Author:	Christian Muenkel					*/
/*		Present address:					*/
/*		muenkel@carbo.tphys.uni-heidelberg.de			*/
/*		Christian Muenkel					*/
/*		Institut fuer Theoretische Physik			*/
/*		Philosophenweg 19					*/
/*		69120 Heidelberg - Germany				*/
/*									*/
/************************************************************************/
/* version      date            comment                                 */
/*                                                                      */
/* V1.0		 1/94		first implementation 			*/
/*				using p3dlib*, IS RayTracer lib ...	*/
/* V1.1		 4/94		filled/shaded Polygons supported 	*/
/*				using z-buffer				*/
/*                                                                      */
/************************************************************************/
/*                                                                      */
/* To do:								*/
/*                                                                      */
/************************************************************************/

#undef		__SGL_MAIN
#include	"sglglob.h"


#define		SGL3D_MATRIX_NUM	64

SGLMatrix	(*ms);

static sgl3d_flag_vismode = SGL3D_PROJ_ORTHO;

#define	DEF_COP_X	0.0
#define	DEF_COP_Y	0.0
#define	DEF_COP_Z	-10.0
#define	DEF_REF_X	0.0
#define	DEF_REF_Y	0.0
#define	DEF_REF_Z	0.0
#define	DEF_VIEW_WX	1.0
#define	DEF_VIEW_WY	1.0
#define	DEF_VIEW_DIST	7.0
#define DEF_CLIP_NEAR	-1.0
#define	DEF_CLIP_FAR	1.0
#define	DEF_CLIP_TOP	1.0
#define	DEF_CLIP_BOT	-1.0
#define	DEF_CLIP_LEFT	-1.0
#define	DEF_CLIP_RIGHT	1.0


static float	gl_sphere_wf[6][8][3];


/************************************************************************/
/************************************************************************/
/*									*/
/*  Initialize SGL3d library						*/
/*									*/
/************************************************************************/
/************************************************************************/
void	sgl3d_init(void)
{
  sgl3d_spherel_init();
  sgl3d_spheresf_init();
}

void	sgl3d_create_win(void)
{
  X11GD3D = (struct X113DGlobalData *) malloc (sizeof(struct X113DGlobalData));
  if (X11GD3D == NULL)
   { fprintf(stderr,"SGL3D_INIT: Can not allocate 3d global data.\n");
     exit;
   }
  X11GD3D_all[GD_act] = X11GD3D;

  X11GD3D->cop_x = DEF_COP_X;
  X11GD3D->cop_y = DEF_COP_Y;
  X11GD3D->cop_z = DEF_COP_Z;
  X11GD3D->ref_x = DEF_REF_X;
  X11GD3D->ref_y = DEF_REF_Y;
  X11GD3D->ref_z = DEF_REF_Z;
  X11GD3D->view_wx = DEF_VIEW_WX;
  X11GD3D->view_wy = DEF_VIEW_WY;
  X11GD3D->vdist = DEF_VIEW_DIST;
  X11GD3D->clip_near = DEF_CLIP_NEAR;
  X11GD3D->clip_far  = DEF_CLIP_FAR;
  X11GD3D->clip_top  = DEF_CLIP_TOP;
  X11GD3D->clip_bot  = DEF_CLIP_BOT;
  X11GD3D->clip_left = DEF_CLIP_LEFT;
  X11GD3D->clip_right= DEF_CLIP_RIGHT;

  X11GD3D->ms = (SGLMatrix *) malloc(SGL3D_MATRIX_NUM*sizeof(SGLMatrix));
  if (X11GD3D->ms == NULL)
   { fprintf(stderr,"SGL3D_INIT: Can not allocate matrix_stack.\n");
     exit;
   }
  X11GD3D->ms_ptr = 0;
  ms = &(X11GD3D->ms[X11GD3D->ms_ptr]);
  sgl3d_matrix_proj_init();

  X11GD3D->zbuf = (float *) calloc(X11GD->win_sx 
		* X11GD->win_sy,sizeof(float));
  if (X11GD3D->zbuf == NULL)
   { fprintf(stderr,"SGL3D_INIT: Can not allocate 3d z-buffer data.\n");
     exit;
   }
  sgl3d_zclear();
  X11GD3D->shading = SGL3D_LINES;
}

void	sgl3d_free_win()
{
  free(X11GD3D_all[GD_act]->zbuf);
  free(X11GD3D_all[GD_act]->ms);
  free(X11GD3D_all[GD_act]);
}

  

void	sgl3d_select_win()
{
 ms = &(X11GD3D->ms[X11GD3D->ms_ptr]);
}

void	sgl3d_select_proj(int mode)
{
   switch(mode)
   { case	SGL3D_PROJ_ORTHO:
	sgl3d_flag_vismode = SGL3D_PROJ_ORTHO;
	break;
     case	SGL3D_PROJ_PERSP:
	sgl3d_flag_vismode = SGL3D_PROJ_PERSP;
	break;
     default:
	fprintf(stderr,"SGL3D_SELECT_PROJ: Unknown projection mode.\n");
	break;
   }
}


/************************************************************************/
/************************************************************************/
/*									*/
/*  Drawing Routines							*/
/*									*/
/************************************************************************/
/************************************************************************/
void	sgl3d_line(float x1,float y1,float z1,float x2,float y2,float z2)
{
  static SGLVector	t1,t2;

  t1[0] = x1; t1[1] = y1; t1[2] = z1; t1[3] = 1.0;
  t2[0] = x2; t2[1] = y2; t2[2] = z2; t2[3] = 1.0;

  sgl3d_vec_trans(t1);
  sgl3d_vec_trans(t2);

  if (t2[2] >= X11GD3D->clip_near && t2[2] <= X11GD3D->clip_far
	&& t1[2] >= X11GD3D->clip_near && t1[2] <= X11GD3D->clip_far)
    XDrawLine(X11GD->disp, X11GD->pm, X11GD->gc,
	(int)t1[0],(int)t1[1],(int)t2[0],(int)t2[1]); 
}

void	sgl3d_circle(float x,float y,float z,float r)
{
  static SGLVector	t,t1;
  float			a;
  int			rs;

  t[0] = x;    t[1] = y;    t[2] = z;    t[3] = 1.0;
  t1[0] = x+r; t1[1] = y+r; t1[2] = z+r; t1[3] = 1.0;

  sgl3d_vec_trans(t);
  sgl3d_vec_trans(t1);
  t1[0] -= t[0]; t1[1] -= t[1]; t1[2] -= t[2]; t1[3] -= t[3];
/*
  a = ((fabs(t1[0])) > (fabs(t1[1]))) ? (fabs(t1[0])):(fabs(t1[1]));
  a = (a > (fabs(t1[2]))) ? a:(fabs(t1[2]));
  rs = a;
*/
  rs = sqrt(t1[0]*t1[0] + t1[1]*t1[1] + t1[2]*t1[2]);

  XFillArc(X11GD->disp,X11GD->pm,X11GD->gc,
	(int)t[0]-rs,(int)t[1]-rs,(int)2*rs,(int)2*rs,0,64*360); 
}

void	sgl3d_polygon(int n, float *x, float *y, float *z)
{
  static SGLVector	t1,t2;
  int			i;

  if (n < 3)
   { fprintf(stderr,"SGL3D_POLYGON: Polygon needs more than 2 sides.\n");
     return;
   }

  t1[0] = x[0];
  t1[1] = y[0];
  t1[2] = z[0];
  t1[3] = 1.0; t2[3] = 1.0;
  sgl3d_vec_trans(t1);
  for (i = n-1; i >= 0; i -=2)
   { t2[0] = x[i];
     t2[1] = y[i];
     t2[2] = z[i];
     sgl3d_vec_trans(t2);
     XDrawLine(X11GD->disp, X11GD->pm, X11GD->gc,
	(int)t1[0],(int)t1[1],(int)t2[0],(int)t2[1]);
     if (i > 0)
      { t1[0] = x[i-1];
        t1[1] = y[i-1];
        t1[2] = z[i-1];
        sgl3d_vec_trans(t1);
        XDrawLine(X11GD->disp, X11GD->pm, X11GD->gc,
		(int)t2[0],(int)t2[1],(int)t1[0],(int)t1[1]);
      }
   }
}

void	sgl3d_spherel(float x, float y, float z, float r)
{
  int			i,j,k;
  static float	tmp_sphere_wf[6][8][4];

  for (j = 5; j >= 0; j--)
    for (i = 7; i >= 0; i--)
      tmp_sphere_wf[j][i][3] = 1.0;
  for (j = 5; j >= 0; j--)
    for (i = 7; i >= 0; i--)
      for (k = 2; k >=0; k--)
        tmp_sphere_wf[j][i][k] = gl_sphere_wf[j][i][k];

   sgl3d_matrix_push();
   sgl3d_trans(x,y,z);
   sgl3d_scale(r,r,r);
   for (j = 5; j >= 0; j--)
    { for (i = 7; i >= 0; i--)
       { sgl3d_vec_trans(tmp_sphere_wf[j][i]);
       }
    }
   sgl3d_matrix_pop();

   for (j = 5; j >= 0; j--)
    { for (i = 6; i >= 0; i--)
       { if (tmp_sphere_wf[j][i][2] >= X11GD3D->clip_near &&
		tmp_sphere_wf[j][i][2] <= X11GD3D->clip_far &&
		tmp_sphere_wf[j][i][2] >= X11GD3D->clip_near &&
		tmp_sphere_wf[j][i][2] <= X11GD3D->clip_far)
           XDrawLine(X11GD->disp, X11GD->pm, X11GD->gc,
		(int)tmp_sphere_wf[j][i][0],(int)tmp_sphere_wf[j][i][1],
		(int)tmp_sphere_wf[j][i+1][0],(int)tmp_sphere_wf[j][i+1][1]); 
       }
      if (tmp_sphere_wf[j][7][2] >= X11GD3D->clip_near &&
		tmp_sphere_wf[j][7][2] <= X11GD3D->clip_far &&
		tmp_sphere_wf[j][0][2] >= X11GD3D->clip_near &&
		tmp_sphere_wf[j][0][2] <= X11GD3D->clip_far)
         XDrawLine(X11GD->disp, X11GD->pm, X11GD->gc,
		(int)tmp_sphere_wf[j][7][0],(int)tmp_sphere_wf[j][7][1],
		(int)tmp_sphere_wf[j][0][0],(int)tmp_sphere_wf[j][0][1]); 
    }
  
}


void	sgl3d_proj_points(int n, float *x,float *y,float *z,
		float *rx, float *ry, float *rz, int *vis)
{
  static SGLVector	tv;
  int			i;

  for (i = n-1; i >= 0; i--)
   { tv[0] = x[i];
     tv[1] = y[i];
     tv[2] = z[i];
     tv[3] = 1.0;
     sgl3d_vec_trans(tv);
     rx[i] = tv[0];
     ry[i] = tv[1];
     rz[i] = tv[2];
     if (tv[2] >= X11GD3D->clip_near && tv[2] <= X11GD3D->clip_far)
	vis[i] = 1;
     else
	vis[i] = -1;
   }
}




/************************************************************************/
/************************************************************************/
/*									*/
/*  Rotation, Movements ...						*/
/*									*/
/************************************************************************/
/************************************************************************/

/************************************************************************/
/* sgl3d_rotx ( float phi )						*/
/*	Rotation around x-axis						*/
/************************************************************************/
void	sgl3d_rotx(float phi)
{
 float		tmp,co,si;

 co = cos(phi); si = sin(phi);
 tmp = (*ms)[1][0];
 (*ms)[1][0] = co*tmp + si*(*ms)[2][0];
 (*ms)[2][0] = co*(*ms)[2][0] - si*tmp;
 tmp = (*ms)[1][1];
 (*ms)[1][1] = co*tmp + si*(*ms)[2][1];
 (*ms)[2][1] = co*(*ms)[2][1] - si*tmp;
 tmp = (*ms)[1][2];
 (*ms)[1][2] = co*tmp + si*(*ms)[2][2];
 (*ms)[2][2] = co*(*ms)[2][2] - si*tmp;
 tmp = (*ms)[1][3];
 (*ms)[1][3] = co*tmp + si*(*ms)[2][3];
 (*ms)[2][3] = co*(*ms)[2][3] - si*tmp;
}

/************************************************************************/
/* sgl3d_roty (float phi)						*/
/*	Rotation around y-axis						*/
/************************************************************************/
void	sgl3d_roty(float phi)
{
 float		tmp,co,si;
 co = cos(phi); si = sin(phi);
 tmp = (*ms)[0][0];
 (*ms)[0][0] = co*tmp - si*(*ms)[2][0];
 (*ms)[2][0] = si*tmp + co*(*ms)[2][0];
 tmp = (*ms)[0][1];
 (*ms)[0][1] = co*tmp - si*(*ms)[2][1];
 (*ms)[2][1] = si*tmp + co*(*ms)[2][1];
 tmp = (*ms)[0][2];
 (*ms)[0][2] = co*tmp - si*(*ms)[2][2];
 (*ms)[2][2] = si*tmp + co*(*ms)[2][2];
 tmp = (*ms)[0][3];
 (*ms)[0][3] = co*tmp - si*(*ms)[2][3];
 (*ms)[2][3] = si*tmp + co*(*ms)[2][3];
}

/************************************************************************/
/* sgl3d_rotz (float phi)						*/
/*	Rotation around z-axis						*/
/************************************************************************/
void	sgl3d_rotz(float phi)
{
 float		tmp,si,co;
 co = cos(phi); si = sin(phi);
 tmp = (*ms)[0][0];
 (*ms)[0][0] = co*tmp + si*(*ms)[1][0];
 (*ms)[1][0] = co*(*ms)[1][0] - si*tmp;
 tmp = (*ms)[0][1];
 (*ms)[0][1] = co*tmp + si*(*ms)[1][1];
 (*ms)[1][1] = co*(*ms)[1][1] - si*tmp;
 tmp = (*ms)[0][2];
 (*ms)[0][2] = co*tmp + si*(*ms)[1][2];
 (*ms)[1][2] = co*(*ms)[1][2] - si*tmp;
 tmp = (*ms)[0][3];
 (*ms)[0][3] = co*tmp + si*(*ms)[1][3];
 (*ms)[1][3] = co*(*ms)[1][3] - si*tmp;
}

/************************************************************************/
/* sgl3d_rot (float x, float y, float z, float ang)			*/
/*	Rotation around vector (d,y,z) , angle ang			*/
/************************************************************************/
void	sgl3d_rot(float x, float y, float z, float ang)
{
  float		ang2,ang3;
  ang2 = atan2(y,x);
  ang3 = atan2(hypot(x,y), z);
  sgl3d_rotz(ang2);		/* remind stack order !	*/
  sgl3d_roty(ang3);
  sgl3d_rotz(ang);
  sgl3d_roty(-ang3);
  sgl3d_rotz(-ang2);		/* this is really the first transf.	*/
}


/************************************************************************/
/* sgl3d_trans (float dx, float dy, float dz)				*/
/*	translate along vector dx,dy,dz					*/
/************************************************************************/
void	sgl3d_trans(float dx, float dy, float dz)
{
  (*ms)[3][0] += (dx * (*ms)[0][0] + dy * (*ms)[1][0] + dz * (*ms)[2][0]);
  (*ms)[3][1] += (dx * (*ms)[0][1] + dy * (*ms)[1][1] + dz * (*ms)[2][1]);
  (*ms)[3][2] += (dx * (*ms)[0][2] + dy * (*ms)[1][2] + dz * (*ms)[2][2]);
  (*ms)[3][3] += (dx * (*ms)[0][3] + dy * (*ms)[1][3] + dz * (*ms)[2][3]);
}

/************************************************************************/
/* sgl3d_scale (float fx, float fy, float fz)				*/
/*	Scale coordinates by factors fx,fy,fz				*/
/************************************************************************/
void	sgl3d_scale(float fx, float fy, float fz)
{
  int		i;
  for (i = 3; i >= 0; i--)  (*ms)[0][i] *= fx;
  for (i = 3; i >= 0; i--)  (*ms)[1][i] *= fy;
  for (i = 3; i >= 0; i--)  (*ms)[2][i] *= fz;
}


/************************************************************************/
/* sgl3d_identity (void)						*/
/*	set temporary matrix to indentity				*/
/************************************************************************/
void	sgl3d_identity(void)
{
 (*ms)[0][0] = 1.0; (*ms)[0][1] = 0.0; (*ms)[0][2] = 0.0; (*ms)[0][3] = 0.0;
 (*ms)[1][0] = 0.0; (*ms)[1][1] = 1.0; (*ms)[1][2] = 0.0; (*ms)[1][3] = 0.0;
 (*ms)[2][0] = 0.0; (*ms)[2][1] = 0.0; (*ms)[2][2] = 1.0; (*ms)[2][3] = 0.0;
 (*ms)[3][0] = 0.0; (*ms)[3][1] = 0.0; (*ms)[3][2] = 0.0; (*ms)[3][3] = 1.0;
}

/************************************************************************/
/************************************************************************/
/*									*/
/*  Matrix, Vector calculations						*/
/*									*/
/************************************************************************/
/************************************************************************/


/************************************************************************/
/* sgl3d_vec_trans (SGLVector v)					*/
/*	Transform Vector v using the Matrix m				*/
/************************************************************************/
void	sgl3d_vec_trans(SGLVector v)
{
  static SGLVector	t;
  t[0] = v[0]*(*ms)[0][0]+v[1]*(*ms)[1][0]+v[2]*(*ms)[2][0]+v[3]*(*ms)[3][0];
  t[1] = v[0]*(*ms)[0][1]+v[1]*(*ms)[1][1]+v[2]*(*ms)[2][1]+v[3]*(*ms)[3][1];
  t[2] = v[0]*(*ms)[0][2]+v[1]*(*ms)[1][2]+v[2]*(*ms)[2][2]+v[3]*(*ms)[3][2];
  t[3] = v[0]*(*ms)[0][3]+v[1]*(*ms)[1][3]+v[2]*(*ms)[2][3]+v[3]*(*ms)[3][3];
  v[0] = t[0]; v[1] = t[1]; v[2] = t[2]; v[3] = t[3];
}
 
/************************************************************************/
/* sgl3d_vec_transpre (SGLVector v)					*/
/*	Transform Vector v using the Matrix m, premultiply		*/
/************************************************************************/
void	sgl3d_vec_transpre(SGLVector v)
{
  static SGLVector	t;
  t[0] = v[0]*(*ms)[0][0]+v[1]*(*ms)[0][1]+v[2]*(*ms)[0][2]+v[3]*(*ms)[0][3];
  t[1] = v[0]*(*ms)[1][0]+v[1]*(*ms)[1][1]+v[2]*(*ms)[1][2]+v[3]*(*ms)[1][3];
  t[2] = v[0]*(*ms)[2][0]+v[1]*(*ms)[2][1]+v[2]*(*ms)[2][2]+v[3]*(*ms)[2][3];
  t[3] = v[0]*(*ms)[3][0]+v[1]*(*ms)[3][1]+v[2]*(*ms)[3][2]+v[3]*(*ms)[3][3];
  v[0] = t[0]; v[1] = t[1]; v[2] = t[2]; v[3] = t[3];
}


/************************************************************************/
/* sgl3d_matrix_mult (SGLMatrix m, SGLMatrix n, SGLMatrix mat)		*/
/*	Matrixmultiplication mat = m * n				*/
/************************************************************************/
void	sgl3d_matrix_mult(SGLMatrix m, SGLMatrix n, SGLMatrix mat)
{
 mat[0][0] = m[0][0] * n[0][0] + m[0][1] * n[1][0] + m[0][2] * n[2][0] + m[0][3] * n[3][0];
 mat[0][1] = m[0][0] * n[0][1] + m[0][1] * n[1][1] + m[0][2] * n[2][1] + m[0][3] * n[3][1];
 mat[0][2] = m[0][0] * n[0][2] + m[0][1] * n[1][2] + m[0][2] * n[2][2] + m[0][3] * n[3][2];
 mat[0][3] = m[0][0] * n[0][3] + m[0][1] * n[1][3] + m[0][2] * n[2][3] + m[0][3] * n[3][3];

 mat[1][0] = m[1][0] * n[0][0] + m[1][1] * n[1][0] + m[1][2] * n[2][0] + m[1][3] * n[3][0];
 mat[1][1] = m[1][0] * n[0][1] + m[1][1] * n[1][1] + m[1][2] * n[2][1] + m[1][3] * n[3][1];
 mat[1][2] = m[1][0] * n[0][2] + m[1][1] * n[1][2] + m[1][2] * n[2][2] + m[1][3] * n[3][2];
 mat[1][3] = m[1][0] * n[0][3] + m[1][1] * n[1][3] + m[1][2] * n[2][3] + m[1][3] * n[3][3];

 mat[2][0] = m[2][0] * n[0][0] + m[2][1] * n[1][0] + m[2][2] * n[2][0] + m[2][3] * n[3][0];
 mat[2][1] = m[2][0] * n[0][1] + m[2][1] * n[1][1] + m[2][2] * n[2][1] + m[2][3] * n[3][1];
 mat[2][2] = m[2][0] * n[0][2] + m[2][1] * n[1][2] + m[2][2] * n[2][2] + m[2][3] * n[3][2];
 mat[2][3] = m[2][0] * n[0][3] + m[2][1] * n[1][3] + m[2][2] * n[2][3] + m[2][3] * n[3][3];
 
 mat[3][0] = m[3][0] * n[0][0] + m[3][1] * n[1][0] + m[3][2] * n[2][0] + m[3][3] * n[3][0];
 mat[3][1] = m[3][0] * n[0][1] + m[3][1] * n[1][1] + m[3][2] * n[2][1] + m[3][3] * n[3][1];
 mat[3][2] = m[3][0] * n[0][2] + m[3][1] * n[1][2] + m[3][2] * n[2][2] + m[3][3] * n[3][2];
 mat[3][3] = m[3][0] * n[0][3] + m[3][1] * n[1][3] + m[3][2] * n[2][3] + m[3][3] * n[3][3];
}

/************************************************************************/
/* sgl3d_matrix_cpy (SGLMatrix m1, SGLMatrix m2)			*/
/*	copy matrix m2 into matrix m1					*/
/************************************************************************/
void	sgl3d_matrix_cpy(SGLMatrix m1, SGLMatrix m2)
{
 memcpy(&m1,&m2,sizeof(SGLMatrix)); 
}


/************************************************************************/
/* sgl3d_matrix_push (void)						*/
/*	Push transformation matrix m onto stack				*/
/************************************************************************/
void	sgl3d_matrix_push(void)
{
  if (X11GD3D->ms_ptr >= SGL3D_MATRIX_NUM)
   { fprintf(stderr,"SGL3D_MATRIX_PUSH: Stack overflow.\n");
   }
  else
   { memcpy((void *)&(X11GD3D->ms[X11GD3D->ms_ptr+1]),
      (const void *)&(X11GD3D->ms[X11GD3D->ms_ptr]),sizeof(SGLMatrix));
     (X11GD3D->ms_ptr)++;
     ms = &(X11GD3D->ms[X11GD3D->ms_ptr]);
   }
}

/************************************************************************/
/* sgl3d_matrix_pop (void)						*/
/*	Pop projection matrix m from stack				*/
/************************************************************************/
void	sgl3d_matrix_pop(void)
{
  if (X11GD3D->ms_ptr <= 0)
   { fprintf(stderr,"SGL3D_MATRIX_PUSH: Stack empty - nothing to pop.\n");
   }
  else
   { (X11GD3D->ms_ptr)--;
     ms = &(X11GD3D->ms[X11GD3D->ms_ptr]);
   }
}


/************************************************************************/
/* sgl3d_matrix_proj_init()						*/
/*	Initialize transformation matrix of projection			*/
/************************************************************************/
void	sgl3d_matrix_proj_init(void)
{
  float		ang2,ang3,d,phi,theta;

  switch(sgl3d_flag_vismode)
   { case SGL3D_PROJ_PERSP:		/* perspective projection*/
	sgl3d_identity();
	break;

     case SGL3D_PROJ_ORTHO:		/* orthogonal projection */
	ang2 = sqrt((X11GD3D->ref_x-X11GD3D->cop_x)*(X11GD3D->ref_x-X11GD3D->cop_x)+(X11GD3D->ref_z-X11GD3D->cop_z)*(X11GD3D->ref_z-X11GD3D->cop_z));
	ang3 = sqrt((X11GD3D->ref_x-X11GD3D->cop_x)*(X11GD3D->ref_x-X11GD3D->cop_x)+(X11GD3D->ref_y-X11GD3D->cop_y)*(X11GD3D->ref_y-X11GD3D->cop_y)+
			(X11GD3D->ref_z-X11GD3D->cop_z)*(X11GD3D->ref_z-X11GD3D->cop_z));
	if (fabs(ang3) > 1e-2)
	 { phi = acos(ang2/ang3);
	   sgl3d_rotx(phi);
	 }
	if (fabs(ang2) > 1e-2)
	 { theta = acos((X11GD3D->cop_z - X11GD3D->ref_z) / ang2);
	   sgl3d_roty(theta);
	 }
	sgl3d_identity();
        d = sqrt((X11GD3D->cop_x-X11GD3D->ref_x)*(X11GD3D->cop_x-X11GD3D->ref_x)+(X11GD3D->cop_y-X11GD3D->ref_y)*(X11GD3D->cop_y-X11GD3D->ref_y)+
		(X11GD3D->cop_z-X11GD3D->ref_z)*(X11GD3D->cop_z-X11GD3D->ref_z));
        X11GD3D->clip_near = - (d - X11GD3D->vdist);
        X11GD3D->clip_far  =   (d - X11GD3D->vdist);
	(*ms)[0][0] = 2.0 / (X11GD3D->clip_right - X11GD3D->clip_left);
	(*ms)[1][1] = 2.0 / (X11GD3D->clip_top - X11GD3D->clip_bot);
	(*ms)[2][2] = -2.0 / (X11GD3D->clip_near - X11GD3D->clip_far);
	(*ms)[3][0] = -(X11GD3D->clip_right+X11GD3D->clip_left) / (X11GD3D->clip_right-X11GD3D->clip_left);
	(*ms)[3][1] = -(X11GD3D->clip_top+X11GD3D->clip_bot) / (X11GD3D->clip_top-X11GD3D->clip_bot);
	(*ms)[3][2] = -(X11GD3D->clip_near+X11GD3D->clip_far) / (X11GD3D->clip_near-X11GD3D->clip_far);
	sgl3d_trans((float)(X11GD->win_sx/2),(float)(X11GD->win_sy/2),0.0);
/*
	sgl3d_scale((float)(win_sx/2),(float)-(win_sy/2),1.0);
	sgl3d_scale(1.0/X11GD3D->view_wx,1.0/X11GD3D->view_wy,1.0);
	sgl3d_trans(-X11GD3D->ref_x,-X11GD3D->ref_y,-X11GD3D->ref_z);
*/
	sgl3d_scale((float)(X11GD->win_sx/2),
		(float)-(X11GD->win_sy/2),(float)X11GD->win_sx);
		sgl3d_scale(1.0/X11GD3D->view_wx,1.0/X11GD3D->view_wy,
		1.0/X11GD3D->view_wx);
	sgl3d_trans(-X11GD3D->ref_x,-X11GD3D->ref_y,-X11GD3D->ref_z);
	X11GD3D->clip_near *= X11GD->win_sx;
	X11GD3D->clip_far  *= X11GD->win_sx;
	break;
     default:
        fprintf(stderr,"SGL3D_INIT: Unknown projection mode.\n");
        break;
   }
}

/************************************************************************/
/* sgl3d_get_cop(float *x, float *y, float *z)				*/
/*	Center of projection						*/
/************************************************************************/
void	sgl3d_get_cop(float *x, float *y, float *z)
{
 *x = X11GD3D->cop_x;
 *y = X11GD3D->cop_y;
 *z = X11GD3D->cop_z;
}


/************************************************************************/
/* sgl3d_set_cop(float x, float y, float z)				*/
/*	Center of projection						*/
/************************************************************************/
void	sgl3d_set_cop(float x, float y, float z)
{
 X11GD3D->cop_x = x;
 X11GD3D->cop_y = y;
 X11GD3D->cop_z = z;
 sgl3d_matrix_proj_init();
}

/************************************************************************/
/* sgl3d_get_ref(float *x, float *y, float *z)				*/
/*	View reference point						*/
/************************************************************************/
void	sgl3d_get_ref(float *x, float *y, float *z)
{
 *x = X11GD3D->ref_x;
 *y = X11GD3D->ref_y;
 *z = X11GD3D->ref_z;
}

/************************************************************************/
/* sgl3d_set_ref(float x, float y, float z)				*/
/*	View reference point						*/
/************************************************************************/
void	sgl3d_set_ref(float x, float y, float z)
{
 X11GD3D->ref_x = x;
 X11GD3D->ref_y = y;
 X11GD3D->ref_z = z;
 sgl3d_matrix_proj_init();
}

/************************************************************************/
/* sgl3d_get_vpar(float *wx, float *wy, float *vd)			*/
/*	View parameters: width/height wx/wy of view window		*/
/*			view distance vd				*/
/************************************************************************/
void	sgl3d_get_vpar(float *wx, float *wy, float *vd)
{
 *wx = X11GD3D->view_wx;
 *wy = X11GD3D->view_wy;
 *vd = X11GD3D->vdist;
}

/************************************************************************/
/* sgl3d_set_vpar(float wx, float wy, float vd)				*/
/*	View parameters: width/height wx/wy of view window		*/
/*			view distance vd				*/
/************************************************************************/
void	sgl3d_set_vpar(float wx, float wy, float vd)
{
 X11GD3D->view_wx = fabs(wx);
 X11GD3D->view_wy = fabs(wy);
 X11GD3D->vdist = vd;
 sgl3d_matrix_proj_init();
}


/************************************************************************/
/* sgl3d_set_shading(char *mode)					*/
/*	Shading mode of 3d library					*/
/*		mode: LINES,GOURAUD,...(?)				*/
/************************************************************************/
void	sgl3d_set_shading(char *mode)
{
  if (strcasecmp(mode,"LINES")==0)
	X11GD3D->shading = SGL3D_LINES;
  else if (strcasecmp(mode,"GOURAUD")==0)
	X11GD3D->shading = SGL3D_GOURAUD;
  else
	X11GD3D->shading = SGL3D_UNKNOWN;
}


/************************************************************************/
/* sgl3d_set_vdefault(void)						*/
/*	set default viewing parameters					*/
/************************************************************************/
void	sgl3d_set_vdefault()
{
  X11GD3D->cop_x = DEF_COP_X;
  X11GD3D->cop_y = DEF_COP_Y;
  X11GD3D->cop_z = DEF_COP_Z;
  X11GD3D->ref_x = DEF_REF_X;
  X11GD3D->ref_y = DEF_REF_Y;
  X11GD3D->ref_z = DEF_REF_Z;
  X11GD3D->view_wx = DEF_VIEW_WX;
  X11GD3D->view_wy = DEF_VIEW_WY;
  X11GD3D->vdist = DEF_VIEW_DIST;
  X11GD3D->clip_near = DEF_CLIP_NEAR;
  X11GD3D->clip_far  = DEF_CLIP_FAR;
  X11GD3D->clip_top  = DEF_CLIP_TOP;
  X11GD3D->clip_bot  = DEF_CLIP_BOT;
  X11GD3D->clip_left = DEF_CLIP_LEFT;
  X11GD3D->clip_right= DEF_CLIP_RIGHT;
  sgl3d_matrix_proj_init();
}


void	sgl3d_spherel_init()
{ 
 int    i;
 float  x,y,z,phi,theta;

 /***** draw wireframe complex sphere *************************************/
 for (i = 0; i < 8; i++)
  { phi = 0;
    theta = i/8.0 * 2.0 * PI;
    x = cos(phi) * sin(theta);
    z = sin(phi) * sin(theta);
    y = cos(theta);
    gl_sphere_wf[0][i][0] = x;
    gl_sphere_wf[0][i][1] = y;
    gl_sphere_wf[0][i][2] = z;
    phi = PI / 4.0;
    theta = i/8.0 * 2.0 * PI;
    x = cos(phi) * sin(theta);
    z = sin(phi) * sin(theta);
    y = cos(theta);
    gl_sphere_wf[1][i][0] = x;
    gl_sphere_wf[1][i][1] = y;
    gl_sphere_wf[1][i][2] = z;
    phi = PI / 2.0;
    theta = i/8.0 * 2.0 * PI;
    x = cos(phi) * sin(theta);
    z = sin(phi) * sin(theta);
    y = cos(theta);
    gl_sphere_wf[2][i][0] = x;
    gl_sphere_wf[2][i][1] = y;
    gl_sphere_wf[2][i][2] = z;
    phi = i/8.0 * 2.0 * PI;
    theta = PI / 4.0;
    x = cos(phi) * sin(theta);
    z = sin(phi) * sin(theta);
    y = cos(theta);
    gl_sphere_wf[3][i][0] = x;
    gl_sphere_wf[3][i][1] = y;
    gl_sphere_wf[3][i][2] = z;
    phi = i/8.0 * 2.0 * PI;
    theta = PI / 2.0;
    x = cos(phi) * sin(theta);
    z = sin(phi) * sin(theta);
    y = cos(theta);
    gl_sphere_wf[4][i][0] = x;
    gl_sphere_wf[4][i][1] = y;
    gl_sphere_wf[4][i][2] = z;
    phi = i/8.0 * 2.0 * PI;
    theta = 3.0 * PI / 4.0;
    x = cos(phi) * sin(theta);
    z = sin(phi) * sin(theta);
    y = cos(theta);
    gl_sphere_wf[5][i][0] = x;
    gl_sphere_wf[5][i][1] = y;
    gl_sphere_wf[5][i][2] = z;
  }
}
