/************************************************************************/
/*									*/
/*	LIBSGL3DSH.C				(C) 1/94 Ch. Muenkel	*/
/*	Three-dimensional extensions to the (basic) SGL library		*/
/*	z-buff / shading is done here					*/
/*									*/
/************************************************************************/
/*									*/
/*	Simple Graphics Library	- 3D					*/
/*	An easy to use interface to the X11 window system		*/
/*      This is the wireframe 3D extension to the SGL library           */
/*									*/
/************************************************************************/
/*									*/
/*	Author:	Christian Muenkel					*/
/*		Present address:					*/
/*		muenkel@carbo.tphys.uni-heidelberg.de			*/
/*		Christian Muenkel					*/
/*		Institut fuer Theoretische Physik			*/
/*		Philosophenweg 19					*/
/*		69120 Heidelberg - Germany				*/
/*									*/
/************************************************************************/
/* version      date            comment                                 */
/*                                                                      */
/* V1.0		 1/94		first implementation 			*/
/*				using p3dlib*, IS RayTracer lib ...	*/
/* V1.1		 4/94		filled/shaded Polygons supported 	*/
/*				using z-buffer				*/
/*				shading added / simple model		*/
/*                                                                      */
/************************************************************************/
/*                                                                      */
/* To do:								*/
/*                                                                      */
/************************************************************************/

#include	"sglglob.h"
#include	"libsgl3dsh.h"


#define         ZMIN                    -32678.0
#define         ZMAX                     32678.99


Poly_vert *poly_dummy;          /* used superficially by POLY_MASK macro */

static void scanline(int , Poly_vert *, Poly_vert *, int );
static incrementalize_y();
static incrementalize_x();
static increment();
static void sgl3d_tri_shade(Poly *);

typedef float	ISVector[3];


static ISVector	light_pos = {0.0, 0.0, -7.0};
static ISVector	light_ambient = { 0.3, 0.3, 0.3 };
static ISVector light_diffuse = { 0.5, 0.5, 0.5 };
static ISVector light_specular = { 0.5, 0.5, 0.5 };
static float	light_spec_bump = 20.0;




/************************************************************************/
/************************************************************************/
/*									*/
/*  Drawing Routines, filled/shaded using z-buffer			*/
/*									*/
/************************************************************************/
/************************************************************************/

void	sgl3d_polyshade(Poly *p)
{
  register int i, li, ri, y, ly, ry, top, rem, mask;
  float ymin;
  Poly_vert l, r, dl, dr;

    ymin = HUGE_VAL;
    for (i=0; i<p->n; i++)              /* find top vertex (y points down) */
        if (p->vert[i].sy < ymin) {
            ymin = p->vert[i].sy;
            top = i;
        }
 
    li = ri = top;                      /* left and right vertex indices */
    rem = p->n;                         /* number of vertices remaining */
    y = ceil(ymin-.5);                  /* current scan line */
    ly = ry = y-1;                      /* lower end of left & right edges */
    mask = p->mask & ~POLY_MASK(sy);    /* stop interpolating screen y */
 
    while (rem>0) {     /* scan in y, activating new edges on left & right */
                        /* as scan line passes over new vertices */
 
        while (ly<=y && rem>0) {        /* advance left edge? */
            rem--;
            i = li-1;                   /* step ccw down left side */
            if (i<0) i = p->n-1;
            incrementalize_y((float *)&p->vert[li], (float *)&p->vert[i],
                    (float *)&l, (float *)&dl, y, mask);
            ly = floor(p->vert[i].sy+.5);
            li = i;
        }
        while (ry<=y && rem>0) {        /* advance right edge? */
            rem--;
            i = ri+1;                   /* step cw down right edge */
            if (i>=p->n) i = 0;
            incrementalize_y((float *)&p->vert[ri], (float *)&p->vert[i],
                    (float *)&r, (float *)&dr, y, mask);
            ry = floor(p->vert[i].sy+.5);
            ri = i;
        }

        while (y<ly && y<ry) {      /* do scanlines till end of l or r edge */
            if (y>=0 && y<=X11GD->win_sy)
                if (l.sx<=r.sx) scanline(y, &l, &r, mask);
                else            scanline(y, &r, &l, mask);
            y++;
            increment((float *)&l, (float *)&dl, mask);
            increment((float *)&r, (float *)&dr, mask);
        }
    }    
}

 
void	sgl3d_zclear()			/* reset z-buffer contents	*/
{
  int		i;
  float	*ptr;

  ptr = X11GD3D->zbuf;
  for (i = (X11GD->win_sx*X11GD->win_sy)-1; i >= 0; i--)
     ptr[i] = ZMAX;
}




/* scanline: output scanline by sampling polygon at Y=y+.5 */
 
static void scanline(int y, Poly_vert *l, Poly_vert *r, int mask)
{
    int x, lx, rx;
    Poly_vert p, dp;
    unsigned long	color;
    float		*ptr;

    ptr = X11GD3D->zbuf;

    mask &= ~POLY_MASK(sx);             /* stop interpolating screen x */
    lx = ceil(l->sx-.5);
    if (lx<0) lx = 0;
    rx = floor(r->sx-.5);
    if (rx>X11GD->win_sx) rx = X11GD->win_sx;
    if (lx>rx) return;
    incrementalize_x((float *)l, (float *)r, (float *)&p, (float *)&dp,
            lx, mask);
    for (x=lx; x<=rx; x++) {            /* scan in x, generating pixels */
        if (p.sz < ptr[y*X11GD->win_sx+x])
         { color = (((unsigned long)(p.r*255)) | 
		((unsigned long)(p.g*255)<<8) |
		((unsigned long)(p.b*255)<<16));
/*
           XSetForeground( X11GD->disp, X11GD->gc, color);
           XDrawPoint(X11GD->disp,X11GD->pm,X11GD->gc,
		x,y);
*/
           XPutPixel(X11GD->xim,x,y,color);
           ptr[y * X11GD->win_sx + x] = p.sz;
         }
        increment((float *)&p, (float *)&dp, mask);
    }
}


/*
 * incrementalize_y: put intersection of line Y=y+.5 with edge between points
 * p1 and p2 in p, put change with respect to y in dp
 */

static incrementalize_y(p1, p2, p, dp, y, mask)
register float *p1, *p2, *p, *dp;
register int mask;
int y;
{
    float dy, frac;

    dy = ((Poly_vert *)p2)->sy - ((Poly_vert *)p1)->sy;
    if (dy==0.) dy = 1.;
    frac = y+.5 - ((Poly_vert *)p1)->sy;

    for (; mask!=0; mask>>=1, p1++, p2++, p++, dp++)
        if (mask&1) {
            *dp = (*p2-*p1)/dy;
            *p = *p1+*dp*frac;
        }
}


/*
 * incrementalize_x: put intersection of line X=x+.5 with edge between points
 * p1 and p2 in p, put change with respect to x in dp
 */
 
static incrementalize_x(p1, p2, p, dp, x, mask)
register float *p1, *p2, *p, *dp;
register int mask;
int x;
{
    float dx, frac;
 
    dx = ((Poly_vert *)p2)->sx - ((Poly_vert *)p1)->sx;
    if (dx==0.) dx = 1.;
    frac = x+.5 - ((Poly_vert *)p1)->sx;
 
    for (; mask!=0; mask>>=1, p1++, p2++, p++, dp++)
        if (mask&1) {
            *dp = (*p2-*p1)/dx;
            *p = *p1+*dp*frac;
        }
}


static increment(p, dp, mask)
register float *p, *dp;
register int mask;
{
    for (; mask!=0; mask>>=1, p++, dp++)
        if (mask&1)
            *p += *dp;
}


void	sgl3d_polygons(int n, float *x, float *y, float *z,
		float *xn, float *yn, float *zn)
{
  Poly	p;
  int	i;

  p.n = n;
  p.mask = POLY_MASK(sx) | POLY_MASK(sy) | POLY_MASK(sz) |
	        POLY_MASK(r) | POLY_MASK(g) | POLY_MASK(b);
  for (i = n-1; i >= 0; i--)
   { p.vert[i].sx = x[i]; 
     p.vert[i].sy = y[i]; 
     p.vert[i].sz = z[i];
     p.vert[i].nx = xn[i];
     p.vert[i].ny = yn[i];
     p.vert[i].nz = zn[i];
   }
  sgl3d_tri_shade(&p);
  sgl3d_polyshade(&p);
}

#ifdef	SIMPLE_SPHERE
static float	tc[6][3] = 
		{{-1.0,0,0},{0,0,-1},{1,0,0},{0,0,1},{0,-1,0},{0,1,0}};
static float	tcn[6][3] = 
		{{-1.0,0,0},{0,0,-1},{1,0,0},{0,0,1},{0,-1,0},{0,1,0}};

static int	tn[8][3] = { {0,1,5},{1,2,5},{2,3,5},{3,0,5},
			{0,4,1},{1,4,2},{2,4,3},{3,4,0} };

void	sgl3d_spheres(float x, float y, float z, float r)
{
   int		i,j;
   static float	tc_tmp[6][4];
   static float	xt[3],yt[3],zt[3];
   static float	xtn[3],ytn[3],ztn[3];

   for (j = 5; j >= 0; j--)
     tc_tmp[j][3] = 1.0;
   for (j = 5; j >= 0; j--)
    for (i = 2; i >= 0; i--)
      tc_tmp[j][i] = tc[j][i];

   sgl3d_matrix_push();
   sgl3d_trans(x,y,z);
   sgl3d_scale(3.0*r,3.0*r,3.0*r);

   for (i = 5; i >= 0; i--)
    sgl3d_vec_trans(tc_tmp[i]);

   sgl3d_matrix_pop();

   for (i = 7; i >= 0; i--)
    { for (j = 2; j >= 0; j--)
       { xt[j] = tc_tmp[tn[i][j]][0];
         yt[j] = tc_tmp[tn[i][j]][1];
         zt[j] = tc_tmp[tn[i][j]][2];
         xtn[j] = tcn[tn[i][j]][0];
         ytn[j] = tcn[tn[i][j]][1];
         ztn[j] = tcn[tn[i][j]][2];
       }
      sgl3d_polygons(3,xt,yt,zt,xtn,ytn,ztn);
    }
}
#endif

#define	SPH_NOD	36
#define	SPH_TRI	68
static float	tc[SPH_NOD][3] = 
		{ {2.1694,0.0000,4.5048},{1.0847,1.8788,4.5048},
		{-1.0847,1.8788,4.5048},{-2.1694,0.0000,4.5048},
		{-1.0847,-1.8788,4.5048},{1.0847,-1.8788,4.5048},
		{3.9092,0.0000,3.1174},{1.9546,3.3854,3.1174},
		{-1.9546,3.3854,3.1174},{-3.9092,0.0000,3.1174},
		{-1.9546,-3.3854,3.1174},{1.9546,-3.3854,3.1174},
		{4.8746,0.0000,1.1126},{2.4373,4.2216,1.1126},
		{-2.4373,4.2216,1.1126},{-4.8746,0.0000,1.1126},
		{-2.4373,-4.2216,1.1126},{2.4373,-4.2216,1.1126},
		{4.8746,0.0000,-1.1126},{2.4373,4.2216,-1.1126},
		{-2.4373,4.2216,-1.1126},{-4.8746,0.0000,-1.1126},
		{-2.4373,-4.2216,-1.1126},{2.4373,-4.2216,-1.1126},
		{3.9092,0.0000,-3.1174},{1.9546,3.3854,-3.1174},
		{-1.9546,3.3854,-3.1174},{-3.9092,0.0000,-3.1174},
		{-1.9546,-3.3854,-3.1174},{1.9546,-3.3854,-3.1174},
		{0.6267,0.0000,4.9606},{0.5427,0.3133,-4.9606},
		{-0.3133,0.5427,4.9606},{-0.6267,-0.0000,-4.9606},
		{-0.3133,-0.5427,4.9606},{0.5427,-0.3133,-4.9606}};

static int	tn[SPH_TRI][3] = 
		{ {0,1,30}, {0,5,11}, {0,5,30}, {0,6,11}, {1,2,7}, {1,2,32},
		{1,6,7}, {1,30,32}, {2,3,8}, {2,3,32}, {2,7,8}, {3,4,9},
		{3,4,34}, {3,8,9}, {3,32,34}, {4,5,10}, {4,5,34}, {4,9,10},
		{5,10,11}, {5,30,34}, {6,7,12}, {6,11,17}, {6,12,17}, {7,8,13},
		{7,12,13}, {8,9,14}, {8,13,14}, {9,10,15}, {9,14,15},
		{10,11,16}, {10,15,16}, {11,16,17}, {12,13,18}, {12,17,23},
		{12,18,23}, {13,14,19}, {13,18,19}, {14,15,20}, {14,19,20},
		{15,16,21}, {15,20,21}, {16,17,22}, {16,21,22}, {17,22,23},
		{18,19,24}, {18,23,29}, {18,24,29}, {19,20,25}, {19,24,25},
		{20,21,26}, {20,25,26}, {21,22,27}, {21,26,27}, {22,23,28},
		{22,27,28}, {23,28,29}, {24,25,31}, {24,29,35}, {24,31,35},
		{25,26,31}, {26,27,33}, {26,31,33}, {27,28,33}, {28,29,35},
		{28,33,35}, {30,32,34}, {31,33,35}  };


void	sgl3d_spheres_init(void)
{
  int	i;
  float	a;
  float	cm[3];

  cm[0] = cm[1] = cm[2] = 0.0;
  for (i = SPH_NOD-1; i >= 0; i--)
   { cm[0] += tc[i][0]; 
     cm[1] += tc[i][1]; 
     cm[2] += tc[i][2]; 
   }
  cm[0] /= SPH_NOD;
  cm[1] /= SPH_NOD;
  cm[2] /= SPH_NOD;
  for (i = SPH_NOD-1; i >= 0; i--)
   { tc[i][0] -= cm[0]; 
     tc[i][1] -= cm[1]; 
     tc[i][2] -= cm[2]; 
   }
  for (i = SPH_NOD-1; i >= 0; i--)
   { a = sqrt(tc[i][0]*tc[i][0]+tc[i][1]*tc[i][1]+tc[i][2]*tc[i][2]);
     tc[i][0] /= a;
     tc[i][1] /= a;
     tc[i][2] /= a;
   }
}


void	sgl3d_spheres(float x, float y, float z, float r)
{
   int		i,j;
   static float	tc_tmp[SPH_NOD][4];
   static float	xt[3],yt[3],zt[3];
   static float	xtn[3],ytn[3],ztn[3];
   float	center[4],a;

   for (j = SPH_NOD-1; j >= 0; j--)
     tc_tmp[j][3] = 1.0;
   for (j = SPH_NOD-1; j >= 0; j--)
    for (i = 2; i >= 0; i--)
      tc_tmp[j][i] = tc[j][i];

   sgl3d_matrix_push();
   sgl3d_trans(x,y,z);
   sgl3d_scale(r,r,r);

   for (i = SPH_NOD-1; i >= 0; i--)
     sgl3d_vec_trans(tc_tmp[i]);
   center[0] = x;
   center[1] = y;
   center[2] = z;
   center[3] = 1.0;
   sgl3d_vec_trans(center);

   sgl3d_matrix_pop();

   for (i = SPH_TRI-1; i >= 0; i--)
    { for (j = 2; j >= 0; j--)
       { xt[j] = tc_tmp[tn[i][j]][0];
         yt[j] = tc_tmp[tn[i][j]][1];
         zt[j] = tc_tmp[tn[i][j]][2];
         xtn[j] = tc_tmp[tn[i][j]][0] - center[0];
         ytn[j] = tc_tmp[tn[i][j]][1] - center[1];
         ztn[j] = tc_tmp[tn[i][j]][2] - center[2];
	 a = sqrt(xtn[j]*xtn[j] + ytn[j]*ytn[j] + ztn[j]*ztn[j]);
         xtn[j] /= a; ytn[j] /= a; ztn[j] /= a;
       }
      sgl3d_polygons(3,xt,yt,zt,xtn,ytn,ztn);
    }
}


/************************************************************************/
/************************************************************************/
/*									*/
/*  Shading / color of triangles					*/
/*									*/
/************************************************************************/
/************************************************************************/
static void	sgl3d_tri_shade(Poly *p)
{
  int	i;
  float	dxl,dyl,dzl,d,cosi,cosn;
  float	dxe,dye,dze,dxh,dyh,dzh;
  float	fac;

 fac = X11GD->win_sx;
 for (i = p->n - 1; i >= 0; i--)
   { dxl = (fac*light_pos[0]) - p->vert[i].sx;
     dyl = (fac*light_pos[1]) - p->vert[i].sy;
     dzl = (fac*light_pos[2]) - p->vert[i].sz;
     d = sqrt(dxl*dxl+dyl*dyl+dzl*dzl);
     dxl /= d; dyl /= d; dzl /= d;
     dxe = X11GD3D->cop_x - p->vert[i].sx;
     dye = X11GD3D->cop_y - p->vert[i].sy;
     dze = X11GD3D->cop_z - p->vert[i].sz;
     d = sqrt(dxe*dxe+dye*dye+dze*dze);
     dxe /= d; dye /= d; dze /= d;
     dxh = dxe+dxl;
     dyh = dye+dyl;
     dzh = dze+dzl;
     d = sqrt(dxh*dxh+dyh*dyh+dzh*dzh);
     dxh /= d; dyh /= d; dzh /= d;
     cosi = fabs(p->vert[i].nx*dxl + p->vert[i].ny*dyl + p->vert[i].nz*dzl);
     cosn = fabs(p->vert[i].nx*dxh + p->vert[i].ny*dyh + p->vert[i].nz*dzh);
     p->vert[i].r = light_ambient[0] + light_diffuse[0] * cosi +
		light_specular[0] * pow(cosn,light_spec_bump); 
     p->vert[i].g = light_ambient[1] + light_diffuse[1] * cosi +
		light_specular[1] * pow(cosn,light_spec_bump); 
     p->vert[i].b = light_ambient[2] + light_diffuse[2] * cosi +
		light_specular[2] * pow(cosn,light_spec_bump); 
   }
}


void	sgl3d_cylinders(float x1, float y1, float z1,
	float x2, float y2, float z2, float r)
{
  int		i,j;
  float		vec1[4],vec2[4],vec3[4],rad,dx,dy,ddx,ddy;
  Poly		p;

  vec1[0] = x1; vec1[1] = y1; vec1[2] = z1; vec1[3] = 1.0;
  sgl3d_vec_trans(vec1);
  vec2[0] = x2; vec2[1] = y2; vec2[2] = z2; vec2[3] = 1.0;
  sgl3d_vec_trans(vec2);
  vec3[0] = x2+r; vec3[1] = y2; vec3[2] = z2; vec3[3] = 1.0;
  sgl3d_vec_trans(vec3);
  vec3[0] -= vec2[0]; vec3[1] -= vec2[1]; vec3[2] -= vec2[2];
  rad = sqrt(vec3[0]*vec3[0]+vec3[1]*vec3[1]+vec3[2]*vec3[2]);

  p.n = 4;
  p.mask = POLY_MASK(sx) | POLY_MASK(sy) | POLY_MASK(sz) |
	        POLY_MASK(r) | POLY_MASK(g) | POLY_MASK(b);

  ddx = (vec1[0]-vec2[0])*(vec1[0]-vec2[0]);
  ddy = (vec1[1]-vec2[1])*(vec1[1]-vec2[1]);
  if (fabs(ddy) > 1E-3)
   { dx = sqrt(rad*rad/(1.0 + ddx/ddy));
     dy = -(dx*(vec2[0]-vec1[0]))/(vec2[1]-vec1[1]);
   }
  else
   { dx = 0.0; dy = rad;
   }

  p.vert[0].sx = vec1[0];
  p.vert[0].sy = vec1[1];
  p.vert[0].sz = vec1[2];
  p.vert[0].r  = 1.0;
  p.vert[0].g  = 1.0;
  p.vert[0].b  = 1.0;

  p.vert[1].sx = vec2[0];
  p.vert[1].sy = vec2[1];
  p.vert[1].sz = vec2[2];
  p.vert[1].r  = 1.0;
  p.vert[1].g  = 1.0;
  p.vert[1].b  = 1.0;

  p.vert[2].sx = vec2[0]+dx;
  p.vert[2].sy = vec2[1]+dy;
  p.vert[2].sz = vec2[2];
  p.vert[2].r  = X11CD->rgb_color[0];
  p.vert[2].g  = X11CD->rgb_color[1];
  p.vert[2].b  = X11CD->rgb_color[2];

  p.vert[3].sx = vec1[0]+dx;
  p.vert[3].sy = vec1[1]+dy;
  p.vert[3].sz = vec1[2];
  p.vert[3].r  = X11CD->rgb_color[0];
  p.vert[3].g  = X11CD->rgb_color[1];
  p.vert[3].b  = X11CD->rgb_color[2];

  sgl3d_polyshade(&p);

  p.vert[0].sx = vec1[0];
  p.vert[0].sy = vec1[1];
  p.vert[0].sz = vec1[2];
  p.vert[0].r  = 1.0;
  p.vert[0].g  = 1.0;
  p.vert[0].b  = 1.0;

  p.vert[1].sx = vec1[0]-dx;
  p.vert[1].sy = vec1[1]-dy;
  p.vert[1].sz = vec1[2];
  p.vert[1].r  = X11CD->rgb_color[0];
  p.vert[1].g  = X11CD->rgb_color[1];
  p.vert[1].b  = X11CD->rgb_color[2];

  p.vert[2].sx = vec2[0]-dx;
  p.vert[2].sy = vec2[1]-dy;
  p.vert[2].sz = vec2[2];
  p.vert[2].r  = X11CD->rgb_color[0];
  p.vert[2].g  = X11CD->rgb_color[1];
  p.vert[2].b  = X11CD->rgb_color[2];

  p.vert[3].sx = vec2[0];
  p.vert[3].sy = vec2[1];
  p.vert[3].sz = vec2[2];
  p.vert[3].r  = 1.0;
  p.vert[3].g  = 1.0;
  p.vert[3].b  = 1.0;

  sgl3d_polyshade(&p);

}


#define	SPHERESF_NUM	8
static float	spheresf_dx[SPHERESF_NUM],spheresf_dy[SPHERESF_NUM];

void	sgl3d_spheresf_init(void)
{
 int	i;

 for (i = SPHERESF_NUM-1; i >= 0; i--)
  { spheresf_dx[i] = cos(2.0*PI*i/SPHERESF_NUM); 
    spheresf_dy[i] = sin(2.0*PI*i/SPHERESF_NUM);
  } 
}


void	sgl3d_spheresf(float x, float y, float z, float r)
{
  int		i,j;
  float		vec1[4],vec2[4],rad;
  Poly		p;

  vec1[0] = x; vec1[1] = y; vec1[2] = z; vec1[3] = 1.0;
  sgl3d_vec_trans(vec1);
  vec2[0] = x+r; vec2[1] = y; vec2[2] = z; vec2[3] = 1.0;
  sgl3d_vec_trans(vec2);
  vec2[0] -= vec1[0]; vec2[1] -= vec1[1]; vec2[2] -= vec1[2];
  rad = sqrt(vec2[0]*vec2[0]+vec2[1]*vec2[1]+vec2[2]*vec2[2]);

  p.n = 3;
  p.mask = POLY_MASK(sx) | POLY_MASK(sy) | POLY_MASK(sz) |
	        POLY_MASK(r) | POLY_MASK(g) | POLY_MASK(b);

  for (i = SPHERESF_NUM-1; i >= 0; i--)
   { p.vert[0].sx = vec1[0];
     p.vert[0].sy = vec1[1];
     p.vert[0].sz = vec1[2]-rad;
     p.vert[0].r  = 1.0;
     p.vert[0].g  = 1.0;
     p.vert[0].b  = 1.0;

     p.vert[1].sx = vec1[0]+rad*spheresf_dx[i];
     p.vert[1].sy = vec1[1]+rad*spheresf_dy[i];
     p.vert[1].sz = vec1[2];
     p.vert[1].r  = X11CD->rgb_color[0];
     p.vert[1].g  = X11CD->rgb_color[1];
     p.vert[1].b  = X11CD->rgb_color[2];

     p.vert[2].sx = vec1[0]+rad*spheresf_dx[(i+1)%SPHERESF_NUM];
     p.vert[2].sy = vec1[1]+rad*spheresf_dy[(i+1)%SPHERESF_NUM];
     p.vert[2].sz = vec1[2];
     p.vert[2].r  = X11CD->rgb_color[0];
     p.vert[2].g  = X11CD->rgb_color[1];
     p.vert[2].b  = X11CD->rgb_color[2];

     sgl3d_polyshade(&p);
   }
}
